<?php
// /home/ubuntu/vtrius_web_system/src/Models/VideoSetting.php

namespace Vtrius\Models;

use Vtrius\Lib\Database;
use PDO;

class VideoSetting {
    private PDO $db;

    public function __construct() {
        $this->db = Database::getConnection();
    }

    /**
     * Create default settings for a newly created video.
     *
     * @param int $videoId
     * @return bool True on success, false on failure.
     */
    public function createDefaults(int $videoId): bool {
        // Check if settings already exist to prevent duplicates
        if ($this->findByVideoId($videoId)) {
            return true; // Already exists, consider it a success
        }

        $sql = "INSERT INTO video_settings (video_id) VALUES (:video_id)";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(":video_id", $videoId, PDO::PARAM_INT);
        return $stmt->execute();
    }

    /**
     * Find settings by video ID.
     *
     * @param int $videoId
     * @return array|false Settings data or false if not found.
     */
    public function findByVideoId(int $videoId) {
        $stmt = $this->db->prepare("SELECT * FROM video_settings WHERE video_id = :video_id");
        $stmt->bindParam(":video_id", $videoId, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetch();
    }

    /**
     * Update video settings.
     *
     * @param int $videoId
     * @param array $settings Associative array of settings to update.
     *                         Keys should match column names in video_settings table.
     *                         Example: ["hide_controls" => 1, "pitch_time_seconds" => 300]
     * @return bool True on success, false on failure.
     */
    public function update(int $videoId, array $settings): bool {
        // Ensure settings exist, create if not (shouldn't happen if createDefaults works)
        if (!$this->findByVideoId($videoId)) {
            if (!$this->createDefaults($videoId)) {
                return false; // Failed to create default settings
            }
        }

        // Build the SET part of the SQL query dynamically
        $setParts = [];
        $params = [":video_id" => $videoId];
        $allowedColumns = [
            "hide_controls", "overlay_image_path", "overlay_opacity", 
            "pause_cover_image_path", "pause_cover_text", 
            "pitch_time_seconds", "pitch_target_selector"
        ];

        foreach ($settings as $key => $value) {
            if (in_array($key, $allowedColumns)) {
                $setParts[] = "{$key} = :{$key}";
                $params[":{$key}"] = $value;
            }
        }

        if (empty($setParts)) {
            return true; // Nothing to update
        }

        $sql = "UPDATE video_settings SET " . implode(", ", $setParts) . " WHERE video_id = :video_id";
        $stmt = $this->db->prepare($sql);

        // Bind parameters
        foreach ($params as $key => &$val) {
            // Determine PDO type (simple check, can be expanded)
            $type = PDO::PARAM_STR;
            if (is_int($val)) {
                $type = PDO::PARAM_INT;
            } elseif (is_bool($val)) {
                $val = (int)$val; // Convert bool to int for SQLite
                $type = PDO::PARAM_INT;
            } elseif (is_float($val)) {
                // PDO doesn't have a specific float type, use STR
                $type = PDO::PARAM_STR;
            } elseif (is_null($val)) {
                $type = PDO::PARAM_NULL;
            }
            $stmt->bindParam($key, $val, $type);
        }

        return $stmt->execute();
    }

    // Note: Deletion is handled by CASCADE constraint when a video is deleted.
}

