<?php
// api/validate.php

// Set header to JSON
header("Content-Type: application/json");

// Include database and functions
require_once __DIR__ . "/../db/database.php"; 
require_once __DIR__ . "/../includes/functions.php";

// Only allow POST requests
if ($_SERVER["REQUEST_METHOD"] !== "POST") {
    http_response_code(405) ; // Method Not Allowed
    echo json_encode(["error" => "Method Not Allowed"]);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents("php://input"), true);

// Validate input
if (!$input || !isset($input["license_key"])) {
    http_response_code(400) ; // Bad Request
    echo json_encode(["error" => "Missing license_key"]);
    exit;
}

$license_key = trim($input["license_key"]);
$site_url = isset($input["site_url"]) ? trim($input["site_url"]) : null;

$response = ["status" => "invalid", "expiry_date" => null];

try {
    $pdo = get_db_connection();
    
    // Find the license
    $stmt = $pdo->prepare("SELECT * FROM licenses WHERE license_key = ?");
    $stmt->execute([$license_key]);
    $license = $stmt->fetch();

    if ($license) {
        // Check validity (this function also updates status if expired)
        $current_status = check_license_validity($pdo, $license["id"]);
        
        // Activation Logic (Activate on first validation if inactive)
        if ($current_status === "inactive") {
            $expiry_date_dt = (new DateTime())->add(new DateInterval("P30D"));
            $expiry_date_str = $expiry_date_dt->format("Y-m-d H:i:s");
            
            $update_stmt = $pdo->prepare("UPDATE licenses SET status = 'active', expiry_date = ?, site_url = ? WHERE id = ?");
            $update_stmt->execute([$expiry_date_str, $site_url, $license["id"]]);
            
            $response["status"] = "active";
            $response["expiry_date"] = $expiry_date_str;
        } else {
            $response["status"] = $current_status;
            if ($license["expiry_date"] !== null) {
                 $response["expiry_date"] = (new DateTime($license["expiry_date"]))->format("Y-m-d H:i:s");
            } else if ($current_status === "active") {
                 $response["expiry_date"] = null; 
            }
        }
        
        // Update site_url if active and different
        if ($current_status === "active" && $site_url && $license["site_url"] !== $site_url) {
             $update_site_stmt = $pdo->prepare("UPDATE licenses SET site_url = ? WHERE id = ?");
             $update_site_stmt->execute([$site_url, $license["id"]]);
        }

    } else {
        $response["status"] = "invalid";
    }

} catch (PDOException $e) {
    http_response_code(500) ; 
    // error_log("Database Error: " . $e->getMessage()); // Uncomment for server-side logging
    $response = ["error" => "An internal error occurred (DB)"];
} catch (Exception $e) {
    http_response_code(500) ;
    // error_log("General Error: " . $e->getMessage()); // Uncomment for server-side logging
    $response = ["error" => "An unexpected error occurred (General)"];
}

// Output the JSON response
echo json_encode($response);
exit;

?>
