<?php
// /home/ubuntu/vtrius_web_system/src/Models/AbTest.php

namespace Vtrius\Models;

use Vtrius\Lib\Database;
use PDO;

class AbTest {
    private PDO $db;

    public function __construct() {
        $this->db = Database::getConnection();
    }

    /**
     * Create a new A/B test.
     *
     * @param int $userId
     * @param string $name
     * @param int $mainVideoId The ID of the video designated as the control.
     * @return int|false The ID of the newly created test or false on failure.
     */
    public function create(int $userId, string $name, int $mainVideoId): int|false {
        $sql = "INSERT INTO ab_tests (user_id, name, main_video_id) VALUES (:user_id, :name, :main_video_id)";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(":user_id", $userId, PDO::PARAM_INT);
        $stmt->bindParam(":name", $name, PDO::PARAM_STR);
        $stmt->bindParam(":main_video_id", $mainVideoId, PDO::PARAM_INT);

        if ($stmt->execute()) {
            $testId = (int)$this->db->lastInsertId();
            // Automatically add the main video as the control variation
            $variationModel = new AbTestVariation();
            $variationModel->addVariation($testId, $mainVideoId, true); // Mark as control
            return $testId;
        } else {
            return false;
        }
    }

    /**
     * Find an A/B test by its ID and user ID (for ownership check).
     *
     * @param int $id
     * @param int $userId
     * @return array|false Test data or false if not found or not owned.
     */
    public function findByIdAndUser(int $id, int $userId): array|false {
        $stmt = $this->db->prepare("SELECT * FROM ab_tests WHERE id = :id AND user_id = :user_id");
        $stmt->bindParam(":id", $id, PDO::PARAM_INT);
        $stmt->bindParam(":user_id", $userId, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetch();
    }

    /**
     * Find all A/B tests for a specific user.
     *
     * @param int $userId
     * @return array List of tests.
     */
    public function findByUser(int $userId): array {
        $stmt = $this->db->prepare("SELECT * FROM ab_tests WHERE user_id = :user_id ORDER BY created_at DESC");
        $stmt->bindParam(":user_id", $userId, PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll();
    }

    /**
     * Update an A/B test (e.g., rename).
     *
     * @param int $id
     * @param int $userId
     * @param string $name
     * @return bool True on success, false on failure.
     */
    public function update(int $id, int $userId, string $name): bool {
        $sql = "UPDATE ab_tests SET name = :name WHERE id = :id AND user_id = :user_id";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(":name", $name, PDO::PARAM_STR);
        $stmt->bindParam(":id", $id, PDO::PARAM_INT);
        $stmt->bindParam(":user_id", $userId, PDO::PARAM_INT);
        return $stmt->execute();
    }

    /**
     * Delete an A/B test and its associated variations and stats.
     * Note: CASCADE constraints handle variation/stat deletion.
     *
     * @param int $id
     * @param int $userId
     * @return bool True on success, false on failure.
     */
    public function delete(int $id, int $userId): bool {
        // First, verify ownership
        if (!$this->findByIdAndUser($id, $userId)) {
            return false; // Test not found or not owned by user
        }

        // Proceed with deletion (CASCADE will handle variations/stats)
        $sql = "DELETE FROM ab_tests WHERE id = :id AND user_id = :user_id";
        $stmt = $this->db->prepare($sql);
        $stmt->bindParam(":id", $id, PDO::PARAM_INT);
        $stmt->bindParam(":user_id", $userId, PDO::PARAM_INT);
        return $stmt->execute();
    }

    // Add other methods as needed, e.g., getting stats summary
}

