<?php
// /home/ubuntu/vtrius_web_system/src/Controllers/VideoController.php

namespace Vtrius\Controllers;

use Vtrius\Lib\Auth;
use Vtrius\Models\Video;
use Vtrius\Models\VideoSetting;
// Placeholder for a future Uploader class
// use Vtrius\Lib\Uploader;

class VideoController {
    private Auth $auth;
    private Video $videoModel;
    private VideoSetting $videoSettingModel;
    private ?int $userId;

    public function __construct() {
        // Ensure config is loaded (includes autoloader and session start)
        // Note: Assuming index.php or a front controller already included config.php
        $this->auth = new Auth();
        $this->auth->requireLogin(BASE_URL . 
        '/login.php
        '); // Ensure user is logged in
        $this->userId = $this->auth->getUserId();
        $this->videoModel = new Video();
        $this->videoSettingModel = new VideoSetting();
    }

    /**
     * Display the list of user's videos.
     */
    public function listVideos(): void {
        $videos = $this->videoModel->findByUser($this->userId);
        $pageTitle = 
        "Meus Vídeos
        "; // PT-BR - Needs i18n later
        $this->loadView(
        'user/videos/list
        ', [
            
            'videos
            ' => $videos,
            
            'pageTitle
            ' => $pageTitle
        ]);
    }

    /**
     * Display the video upload form.
     */
    public function showUploadForm(): void {
        $pageTitle = 
        "Carregar Novo Vídeo
        "; // PT-BR - Needs i18n later
        $this->loadView(
        'user/videos/upload
        ', [
            
            'pageTitle
            ' => $pageTitle
        ]);
    }

    /**
     * Handle the video upload process.
     */
    public function handleUpload(): void {
        if ($_SERVER[
        'REQUEST_METHOD
        '] !== 
        'POST
        ') {
            http_response_code(405);
            echo 
            "Método não permitido
            "; // PT-BR - Needs i18n later
            exit;
        }

        $title = $_POST[
        'title
        '] ?? 
        'Vídeo sem título
        '; // PT-BR - Needs i18n later
        $videoFile = $_FILES[
        'video_file
        '] ?? null;

        // --- Basic Validation ---
        if (!$videoFile || $videoFile[
        'error
        '] !== UPLOAD_ERR_OK) {
            $errorMessage = $this->getUploadErrorMessage($videoFile[
            'error
            '] ?? UPLOAD_ERR_NO_FILE);
            $this->flashMessage(
            "Erro no upload: 
            " . $errorMessage, 
            'error
            '); // PT-BR - Needs i18n later
            $this->redirect(BASE_URL . 
            '/user/videos/upload.php
            ');
        }

        // Validate file type (using MIME type)
        $finfo = new \finfo(FILEINFO_MIME_TYPE);
        $mimeType = $finfo->file($videoFile[
        'tmp_name
        ']);
        if (!in_array($mimeType, ALLOWED_VIDEO_MIMES)) {
            $this->flashMessage(
            "Erro: Tipo de arquivo de vídeo não permitido.
            ", 
            'error
            '); // PT-BR - Needs i18n later
            $this->redirect(BASE_URL . 
            '/user/videos/upload.php
            ');
        }

        // Validate file size
        $maxSizeBytes = MAX_UPLOAD_SIZE_MB * 1024 * 1024;
        if ($videoFile[
        'size
        '] > $maxSizeBytes) {
            $this->flashMessage(
            "Erro: Arquivo de vídeo excede o tamanho máximo permitido (
            " . MAX_UPLOAD_SIZE_MB . 
            " MB).
            ", 
            'error
            '); // PT-BR - Needs i18n later
            $this->redirect(BASE_URL . 
            '/user/videos/upload.php
            ');
        }

        // --- File Handling ---
        $originalFilename = basename($videoFile[
        'name
        ']);
        $extension = pathinfo($originalFilename, PATHINFO_EXTENSION);
        // Create a unique filename to prevent collisions
        $uniqueFilename = uniqid(
        'video_
        ', true) . 
        '.
        ' . strtolower($extension);
        
        // User-specific upload directory
        $userUploadDir = VIDEOS_DIR . 
        '/
        ' . $this->userId;
        if (!is_dir($userUploadDir)) {
            if (!mkdir($userUploadDir, 0755, true)) {
                $this->flashMessage(
                "Erro: Não foi possível criar o diretório de upload.
                ", 
                'error
                '); // PT-BR - Needs i18n later
                $this->redirect(BASE_URL . 
                '/user/videos/upload.php
                ');
            }
        }

        $destinationPath = $userUploadDir . 
        '/
        ' . $uniqueFilename;
        $relativePath = $this->userId . 
        '/
        ' . $uniqueFilename; // Path to store in DB

        // Move the uploaded file
        if (move_uploaded_file($videoFile[
        'tmp_name
        '], $destinationPath)) {
            // --- Database Record Creation ---
            $videoId = $this->videoModel->create(
                $this->userId,
                $title,
                $originalFilename,
                $relativePath, // Store relative path
                $mimeType,
                $videoFile[
                'size
                ']
            );

            if ($videoId) {
                $this->flashMessage(
                "Vídeo '
                " . htmlspecialchars($title) . 
                "' carregado com sucesso!
                ", 
                'success
                '); // PT-BR - Needs i18n later
                $this->redirect(BASE_URL . 
                '/user/videos.php
                '); // Redirect to video list
            } else {
                // Database error, delete the uploaded file
                @unlink($destinationPath);
                $this->flashMessage(
                "Erro: Não foi possível salvar as informações do vídeo no banco de dados.
                ", 
                'error
                '); // PT-BR - Needs i18n later
                $this->redirect(BASE_URL . 
                '/user/videos/upload.php
                ');
            }
        } else {
            $this->flashMessage(
            "Erro: Falha ao mover o arquivo carregado para o destino.
            ", 
            'error
            '); // PT-BR - Needs i18n later
            $this->redirect(BASE_URL . 
            '/user/videos/upload.php
            ');
        }
    }

    /**
     * Handle video deletion.
     */
    public function handleDelete(): void {
        if ($_SERVER[
        'REQUEST_METHOD
        '] !== 
        'POST
        ' || !isset($_POST[
        'video_id
        '])) {
            http_response_code(400);
            echo 
            "Requisição inválida
            "; // PT-BR - Needs i18n later
            exit;
        }

        $videoId = filter_input(INPUT_POST, 
        'video_id
        ', FILTER_VALIDATE_INT);

        if (!$videoId) {
             http_response_code(400);
            echo 
            "ID de vídeo inválido
            "; // PT-BR - Needs i18n later
            exit;
        }

        // delete() method in Video model already checks ownership and deletes file
        if ($this->videoModel->delete($videoId, $this->userId)) {
            $this->flashMessage(
            "Vídeo excluído com sucesso.
            ", 
            'success
            '); // PT-BR - Needs i18n later
        } else {
            $this->flashMessage(
            "Erro ao excluir o vídeo. Ele pode não existir ou você não tem permissão.
            ", 
            'error
            '); // PT-BR - Needs i18n later
        }
        $this->redirect(BASE_URL . 
        '/user/videos.php
        ');
    }
    
    // --- Helper Methods ---

    /**
     * Simple view loader (can be expanded).
     */
    private function loadView(string $viewName, array $data = []): void {
        extract($data);
        $viewPath = APP_ROOT . 
        '/Views/
        ' . $viewName . 
        '.php
        ';
        if (file_exists($viewPath)) {
            // Include user-specific layout or a general one
            $headerPath = APP_ROOT . 
            '/Views/layouts/user_header.php
            '; // Assuming a user-specific header
            $footerPath = APP_ROOT . 
            '/Views/layouts/user_footer.php
            '; // Assuming a user-specific footer
            
            if (!file_exists($headerPath)) $headerPath = APP_ROOT . 
            '/Views/layouts/header.php
            '; // Fallback
            if (!file_exists($footerPath)) $footerPath = APP_ROOT . 
            '/Views/layouts/footer.php
            '; // Fallback

            include $headerPath;
            include $viewPath;
            include $footerPath;
        } else {
            http_response_code(500);
            echo 
            "Erro: View não encontrada ({$viewPath})
            "; // PT-BR - Needs i18n later
        }
    }

    /**
     * Redirect to a given URL.
     */
    private function redirect(string $url): void {
        header(
        'Location: 
        ' . $url);
        exit;
    }

    /**
     * Set a flash message in the session.
     */
    private function flashMessage(string $message, string $type = 
    'success
    '): void {
        // Ensure session is active
        if (session_status() == PHP_SESSION_NONE) {
            session_start();
        }
        $_SESSION[
        'flash_message
        '] = [
        'message
        ' => $message, 
        'type
        ' => $type];
    }

    /**
     * Get a user-friendly error message for upload errors.
     */
    private function getUploadErrorMessage(int $errorCode): string {
        return match ($errorCode) {
            UPLOAD_ERR_INI_SIZE => 
            'O arquivo excede a diretiva upload_max_filesize no php.ini.
            ', // PT-BR
            UPLOAD_ERR_FORM_SIZE => 
            'O arquivo excede a diretiva MAX_FILE_SIZE especificada no formulário HTML.
            ', // PT-BR
            UPLOAD_ERR_PARTIAL => 
            'O upload do arquivo foi feito parcialmente.
            ', // PT-BR
            UPLOAD_ERR_NO_FILE => 
            'Nenhum arquivo foi enviado.
            ', // PT-BR
            UPLOAD_ERR_NO_TMP_DIR => 
            'Faltando uma pasta temporária.
            ', // PT-BR
            UPLOAD_ERR_CANT_WRITE => 
            'Falha ao escrever o arquivo em disco.
            ', // PT-BR
            UPLOAD_ERR_EXTENSION => 
            'Uma extensão do PHP interrompeu o upload do arquivo.
            ', // PT-BR
            default => 
            'Erro desconhecido no upload.
            ', // PT-BR
        };
    }
}

