<?php
// /home/ubuntu/vtrius_web_system/src/Lib/Auth.php

namespace Vtrius\Lib;

use Vtrius\Models\User;
use DateTime;

class Auth {
    private User $userModel;

    public function __construct() {
        $this->userModel = new User();
        // Ensure session is started whenever Auth is instantiated
        if (session_status() == PHP_SESSION_NONE) {
            session_start();
        }
    }

    /**
     * Attempt to log in a user.
     *
     * @param string $username
     * @param string $password
     * @return bool True on successful login, false otherwise.
     */
    public function login(string $username, string $password): bool {
        $user = $this->userModel->findByUsername($username);

        if ($user && password_verify($password, $user["password_hash"])) {
            // Check if the account is active and not expired
            if ($this->isUserAccountActive($user)) {
                // Regenerate session ID to prevent session fixation
                session_regenerate_id(true);

                // Store user data in session
                $_SESSION["user_id"] = $user["id"];
                $_SESSION["username"] = $user["username"];
                $_SESSION["is_admin"] = (bool)$user["is_admin"];
                $_SESSION["login_time"] = time(); 

                // Update expired status if needed (could be done elsewhere too)
                $this->userModel->updateExpiredUsers();

                return true;
            }
        }
        return false;
    }

    /**
     * Log out the current user.
     */
    public function logout(): void {
        // Unset all session variables
        $_SESSION = [];

        // Destroy the session cookie
        if (ini_get("session.use_cookies")) {
            $params = session_get_cookie_params();
            setcookie(session_name(), 
                      ", 
                      time() - 42000,
                      $params["path"], 
                      $params["domain"],
                      $params["secure"], 
                      $params["httponly"]
            );
        }

        // Destroy the session
        session_destroy();
    }

    /**
     * Check if a user is currently logged in.
     *
     * @return bool True if logged in, false otherwise.
     */
    public function isLoggedIn(): bool {
        return isset($_SESSION["user_id"]);
    }

    /**
     * Get the currently logged-in user ID.
     *
     * @return int|null User ID or null if not logged in.
     */
    public function getUserId(): ?int {
        return $_SESSION["user_id"] ?? null;
    }

    /**
     * Get the currently logged-in username.
     *
     * @return string|null Username or null if not logged in.
     */
    public function getUsername(): ?string {
        return $_SESSION["username"] ?? null;
    }

    /**
     * Check if the currently logged-in user is an admin.
     *
     * @return bool True if admin, false otherwise.
     */
    public function isAdmin(): bool {
        return isset($_SESSION["is_admin"]) && $_SESSION["is_admin"] === true;
    }

    /**
     * Check if a user account is currently active (status is active and not expired).
     *
     * @param array $user User data array from the database.
     * @return bool True if the account is active, false otherwise.
     */
    public function isUserAccountActive(array $user): bool {
        if ($user["status"] !== "active") {
            return false;
        }
        // Admins or users with no expiry date are always considered active if status is 'active'
        if ($user["is_admin"] || $user["expiry_date"] === null) {
            return true;
        }
        // Check expiry date for non-admin users
        try {
            $expiryDateTime = new DateTime($user["expiry_date"]);
            return new DateTime() <= $expiryDateTime;
        } catch (\Exception $e) {
            // Invalid date format, treat as inactive/expired for safety
            error_log("Invalid expiry date format for user ID: " . $user["id"]);
            return false;
        }
    }

    /**
     * Require the user to be logged in. Redirects to login page if not.
     *
     * @param string $redirectUrl The URL to redirect to if not logged in.
     */
    public function requireLogin(string $redirectUrl = 
    "/login.php"): void {
        if (!$this->isLoggedIn()) {
            // Maybe store the intended destination URL in session to redirect back after login
            // $_SESSION["redirect_after_login"] = $_SERVER["REQUEST_URI"];
            header("Location: " . $redirectUrl);
            exit;
        }
        // Additionally, re-check if the logged-in user is still active
        $user = $this->userModel->findById($this->getUserId());
        if (!$user || !$this->isUserAccountActive($user)) {
            $this->logout(); // Log out inactive/expired user
            header("Location: " . $redirectUrl . "?error=account_inactive");
            exit;
        }
    }

    /**
     * Require the user to be an admin. Redirects if not.
     *
     * @param string $redirectUrl The URL to redirect to if not an admin.
     */
    public function requireAdmin(string $redirectUrl = "/dashboard.php"): void {
        $this->requireLogin(); // First ensure user is logged in and active
        if (!$this->isAdmin()) {
            header("Location: " . $redirectUrl . "?error=access_denied");
            exit;
        }
    }
}

