<?php
// /home/ubuntu/vtrius_web_system/src/Controllers/AbTestController.php

namespace Vtrius\Controllers;

use Vtrius\Lib\Auth;
use Vtrius\Models\AbTest;
use Vtrius\Models\AbTestVariation;
use Vtrius\Models\Video;

class AbTestController {
    private Auth $auth;
    private AbTest $abTestModel;
    private AbTestVariation $abVariationModel;
    private Video $videoModel;
    private ?int $userId;

    public function __construct() {
        // Ensure config is loaded
        $this->auth = new Auth();
        $this->auth->requireLogin(BASE_URL . "/login.php");
        $this->userId = $this->auth->getUserId();
        $this->abTestModel = new AbTest();
        $this->abVariationModel = new AbTestVariation();
        $this->videoModel = new Video(); // Needed to list videos for selection
    }

    /**
     * Display the list of user's A/B tests.
     */
    public function listTests(): void {
        $tests = $this->abTestModel->findByUser($this->userId);
        $pageTitleKey = "ab_tests_list_title"; // Needs translation
        $this->loadView("user/abtests/list", [
            "pageTitleKey" => $pageTitleKey,
            "tests" => $tests
        ]);
    }

    /**
     * Display the form to create a new A/B test.
     */
    public function showCreateForm(): void {
        // Get user's videos to populate the control video selection
        $videos = $this->videoModel->findByUser($this->userId);
        if (empty($videos)) {
            $this->flashMessage(__("ab_test_error_no_videos"), "warning"); // Needs translation
            $this->redirect(BASE_URL . "/user/videos/upload.php");
        }

        $pageTitleKey = "ab_test_create_title"; // Needs translation
        $this->loadView("user/abtests/create", [
            "pageTitleKey" => $pageTitleKey,
            "videos" => $videos
        ]);
    }

    /**
     * Handle the creation of a new A/B test.
     */
    public function handleCreate(): void {
        if ($_SERVER["REQUEST_METHOD"] !== "POST") {
            http_response_code(405);
            echo __("method_not_allowed");
            exit;
        }

        $name = trim($_POST["name"] ?? "");
        $mainVideoId = filter_input(INPUT_POST, "main_video_id", FILTER_VALIDATE_INT);

        if (empty($name) || !$mainVideoId) {
            $this->flashMessage(__("ab_test_error_missing_fields"), "error"); // Needs translation
            $this->redirect(BASE_URL . "/user/abtests/create.php");
        }

        // Verify ownership of the selected main video
        $video = $this->videoModel->findByIdAndUser($mainVideoId, $this->userId);
        if (!$video) {
            $this->flashMessage(__("ab_test_error_video_not_owned"), "error"); // Needs translation
            $this->redirect(BASE_URL . "/user/abtests/create.php");
        }

        $testId = $this->abTestModel->create($this->userId, $name, $mainVideoId);

        if ($testId) {
            $this->flashMessage(__("ab_test_create_success", ["name" => $name]), "success"); // Needs translation
            $this->redirect(BASE_URL . "/user/abtests/view.php?id=" . $testId); // Redirect to view/edit the new test
        } else {
            $this->flashMessage(__("ab_test_create_error"), "error"); // Needs translation
            $this->redirect(BASE_URL . "/user/abtests/create.php");
        }
    }

    /**
     * Display the details and variations of a specific A/B test.
     */
    public function viewTest(): void {
        $testId = filter_input(INPUT_GET, "id", FILTER_VALIDATE_INT);

        if (!$testId) {
            $this->flashMessage(__("invalid_request"), "error");
            $this->redirect(BASE_URL . "/user/abtests.php");
        }

        // Verify test ownership
        $test = $this->abTestModel->findByIdAndUser($testId, $this->userId);
        if (!$test) {
            $this->flashMessage(__("ab_test_error_not_found"), "error"); // Needs translation
            $this->redirect(BASE_URL . "/user/abtests.php");
        }

        $variations = $this->abVariationModel->findByTestId($testId);
        $userVideos = $this->videoModel->findByUser($this->userId); // For adding new variations
        
        // TODO: Fetch stats for the variations
        $stats = []; // Placeholder

        $pageTitleKey = "ab_test_view_title"; // Needs translation
        $this->loadView("user/abtests/view", [
            "pageTitleKey" => $pageTitleKey,
            "test" => $test,
            "variations" => $variations,
            "userVideos" => $userVideos,
            "stats" => $stats
        ]);
    }

    /**
     * Handle adding a new variation to an A/B test.
     */
    public function handleAddVariation(): void {
        if ($_SERVER["REQUEST_METHOD"] !== "POST") {
            http_response_code(405);
            echo __("method_not_allowed");
            exit;
        }

        $testId = filter_input(INPUT_POST, "test_id", FILTER_VALIDATE_INT);
        $videoId = filter_input(INPUT_POST, "video_id", FILTER_VALIDATE_INT);

        if (!$testId || !$videoId) {
            $this->flashMessage(__("invalid_request"), "error");
            $this->redirect(BASE_URL . "/user/abtests.php");
        }

        // Verify test ownership
        $test = $this->abTestModel->findByIdAndUser($testId, $this->userId);
        if (!$test) {
            $this->flashMessage(__("ab_test_error_not_found"), "error");
            $this->redirect(BASE_URL . "/user/abtests.php");
        }

        // Verify video ownership
        $video = $this->videoModel->findByIdAndUser($videoId, $this->userId);
        if (!$video) {
            $this->flashMessage(__("ab_test_error_video_not_owned"), "error");
            $this->redirect(BASE_URL . "/user/abtests/view.php?id=" . $testId);
        }

        // Attempt to add variation (model checks limit and duplicates)
        $result = $this->abVariationModel->addVariation($testId, $videoId);

        if ($result) {
            $this->flashMessage(__("ab_test_variation_add_success"), "success"); // Needs translation
        } else {
            // Check count to provide specific error message
            $stmtCount = Database::getConnection()->prepare("SELECT COUNT(*) FROM ab_variations WHERE ab_test_id = :ab_test_id");
            $stmtCount->bindParam(":ab_test_id", $testId, PDO::PARAM_INT);
            $stmtCount->execute();
            $count = $stmtCount->fetchColumn();
            if ($count >= 4) {
                 $this->flashMessage(__("ab_test_error_variation_limit"), "error"); // Needs translation
            } else {
                 $this->flashMessage(__("ab_test_variation_add_error"), "error"); // Needs translation (e.g., already exists)
            }
        }

        $this->redirect(BASE_URL . "/user/abtests/view.php?id=" . $testId);
    }

    /**
     * Handle removing a variation from an A/B test.
     */
    public function handleRemoveVariation(): void {
         if ($_SERVER["REQUEST_METHOD"] !== "POST") {
            http_response_code(405);
            echo __("method_not_allowed");
            exit;
        }

        $variationId = filter_input(INPUT_POST, "variation_id", FILTER_VALIDATE_INT);
        $testId = filter_input(INPUT_POST, "test_id", FILTER_VALIDATE_INT); // For redirect and ownership check

        if (!$variationId || !$testId) {
            $this->flashMessage(__("invalid_request"), "error");
            $this->redirect(BASE_URL . "/user/abtests.php");
        }
        
        // Verify test ownership first
        $test = $this->abTestModel->findByIdAndUser($testId, $this->userId);
        if (!$test) {
            $this->flashMessage(__("ab_test_error_not_found"), "error");
            $this->redirect(BASE_URL . "/user/abtests.php");
        }

        // Attempt to remove (model prevents removing control)
        if ($this->abVariationModel->removeVariation($variationId, $testId)) {
            $this->flashMessage(__("ab_test_variation_remove_success"), "success"); // Needs translation
        } else {
            $this->flashMessage(__("ab_test_variation_remove_error"), "error"); // Needs translation (e.g., is control)
        }

        $this->redirect(BASE_URL . "/user/abtests/view.php?id=" . $testId);
    }
    
    /**
     * Handle deleting an entire A/B test.
     */
    public function handleDeleteTest(): void {
         if ($_SERVER["REQUEST_METHOD"] !== "POST") {
            http_response_code(405);
            echo __("method_not_allowed");
            exit;
        }
        
        $testId = filter_input(INPUT_POST, "test_id", FILTER_VALIDATE_INT);
        
        if (!$testId) {
            $this->flashMessage(__("invalid_request"), "error");
            $this->redirect(BASE_URL . "/user/abtests.php");
        }
        
        // Model handles ownership check
        if ($this->abTestModel->delete($testId, $this->userId)) {
             $this->flashMessage(__("ab_test_delete_success"), "success"); // Needs translation
        } else {
             $this->flashMessage(__("ab_test_delete_error"), "error"); // Needs translation
        }
        
        $this->redirect(BASE_URL . "/user/abtests.php");
    }

    // --- Helper Methods (Copied from VideoSettingsController, consider a BaseController) ---

    private function loadView(string $viewName, array $data = []): void {
        extract($data);
        $viewPath = APP_ROOT . "/Views/" . $viewName . ".php";
        if (file_exists($viewPath)) {
            $headerPath = APP_ROOT . "/Views/layouts/user_header.php";
            $footerPath = APP_ROOT . "/Views/layouts/user_footer.php";
            include $headerPath;
            include $viewPath;
            include $footerPath;
        } else {
            http_response_code(500);
            echo __("view_not_found", ["path" => $viewPath]);
        }
    }

    private function redirect(string $url): void {
        header("Location: " . $url);
        exit;
    }

    private function flashMessage(string $messageOrKey, string $type = "success", array $params = []): void {
        if (session_status() == PHP_SESSION_NONE) {
            session_start();
        }
        $_SESSION["flash_message"] = ["key" => $messageOrKey, "params" => $params, "type" => $type];
    }
}

