// /home/ubuntu/vtrius_web_system/public/assets/js/vtrius-player.js

class VtriusPlayer {
    constructor(containerId, options) {
        this.container = document.getElementById(containerId);
        if (!this.container) {
            console.error(`VtriusPlayer Error: Container #${containerId} not found.`);
            return;
        }

        this.options = Object.assign({
            videoId: null,
            videoUrl: "", // Direct URL to the video file
            hideControls: false,
            autoplay: false,
            overlayImageUrl: null,
            overlayOpacity: 1.0,
            pauseCoverImageUrl: null,
            pauseCoverText: "Você já estava assistindo...", // Default text, needs i18n
            pitchTimeSeconds: null,
            pitchTargetSelector: null,
            pitchButtonText: null, // Added
            pitchButtonUrl: null, // Added
            // Add more options as needed (e.g., translations for buttons)       }, options);

        if (!this.options.videoUrl) {
            console.error(`VtriusPlayer Error: videoUrl option is required.`);
            return;
        }

        this.videoElement = null;
        this.overlayElement = null;
        this.pauseCoverElement = null;
        this.pitchTriggered = false;

        this.initPlayer();
    }

    initPlayer() {
        this.container.style.position = "relative"; // Needed for overlays
        this.container.style.overflow = "hidden"; // Prevent content spill

        // Create video element
        this.videoElement = document.createElement("video");
        this.videoElement.src = this.options.videoUrl;
        this.videoElement.style.width = "100%";
        this.videoElement.style.height = "auto";
        this.videoElement.style.display = "block";
        this.videoElement.playsInline = true; // Important for mobile

        if (this.options.hideControls) {
            this.videoElement.controls = false;
        } else {
            this.videoElement.controls = true;
        }

        if (this.options.autoplay) {
            this.videoElement.autoplay = true;
            this.videoElement.muted = true; // Autoplay often requires mute initially
            // Consider adding an unmute button or interaction
        }

        this.container.appendChild(this.videoElement);

        // Create overlay if specified
        if (this.options.overlayImageUrl) {
            this.createOverlay();
        }

        // Create pause cover if specified
        if (this.options.pauseCoverImageUrl) {
            this.createPauseCover();
        }

        // Add event listeners
        this.addEventListeners();

        console.log(`VtriusPlayer initialized for container #${this.container.id}`);
    }

    createOverlay() {
        this.overlayElement = document.createElement("div");
        this.overlayElement.style.position = "absolute";
        this.overlayElement.style.top = "0";
        this.overlayElement.style.left = "0";
        this.overlayElement.style.width = "100%";
        this.overlayElement.style.height = "100%";
        this.overlayElement.style.backgroundImage = `url("${this.options.overlayImageUrl}")`;
        this.overlayElement.style.backgroundSize = "cover"; // Or contain, based on preference
        this.overlayElement.style.backgroundPosition = "center";
        this.overlayElement.style.opacity = this.options.overlayOpacity;
        this.overlayElement.style.pointerEvents = "none"; // Allow clicks to pass through to video
        this.overlayElement.style.zIndex = "10"; // Ensure it's above the video
        this.container.appendChild(this.overlayElement);
    }

    createPauseCover() {
        this.pauseCoverElement = document.createElement("div");
        this.pauseCoverElement.style.position = "absolute";
        this.pauseCoverElement.style.top = "0";
        this.pauseCoverElement.style.left = "0";
        this.pauseCoverElement.style.width = "100%";
        this.pauseCoverElement.style.height = "100%";
        this.pauseCoverElement.style.backgroundImage = `url("${this.options.pauseCoverImageUrl}")`;
        this.pauseCoverElement.style.backgroundSize = "cover";
        this.pauseCoverElement.style.backgroundPosition = "center";
        this.pauseCoverElement.style.backgroundColor = "rgba(0, 0, 0, 0.7)"; // Dark overlay
        this.pauseCoverElement.style.display = "none"; // Hidden by default
        this.pauseCoverElement.style.zIndex = "20";
        this.pauseCoverElement.style.color = "white";
        this.pauseCoverElement.style.textAlign = "center";
        this.pauseCoverElement.style.paddingTop = "20%"; // Adjust as needed

        // Add text
        const textElement = document.createElement("p");
        textElement.textContent = this.options.pauseCoverText; // Needs i18n
        textElement.style.fontSize = "1.5em";
        textElement.style.marginBottom = "20px";
        this.pauseCoverElement.appendChild(textElement);

        // Add buttons
        const buttonContainer = document.createElement("div");

        const continueButton = document.createElement("button");
        continueButton.textContent = "Continuar de onde parou"; // Needs i18n
        continueButton.style.margin = "0 10px";
        continueButton.style.padding = "10px 20px";
        continueButton.onclick = () => this.resumeVideo();
        buttonContainer.appendChild(continueButton);

        const restartButton = document.createElement("button");
        restartButton.textContent = "Reiniciar do início"; // Needs i18n
        restartButton.style.margin = "0 10px";
        restartButton.style.padding = "10px 20px";
        restartButton.onclick = () => this.restartVideo();
        buttonContainer.appendChild(restartButton);

        this.pauseCoverElement.appendChild(buttonContainer);
        this.container.appendChild(this.pauseCoverElement);
    }

    addEventListeners() {
        if (!this.videoElement) return;

        // Show pause cover on pause
        this.videoElement.addEventListener("pause", () => {
            // Don't show cover if video ended or seeking
            if (this.pauseCoverElement && !this.videoElement.ended && !this.videoElement.seeking) {
                this.pauseCoverElement.style.display = "block";
            }
        });

        // Hide pause cover on play
        this.videoElement.addEventListener("play", () => {
            if (this.pauseCoverElement) {
                this.pauseCoverElement.style.display = "none";
            }
        });

        // Pitch container logic
        if (this.options.pitchTimeSeconds !== null && this.options.pitchTargetSelector) {
            this.videoElement.addEventListener("timeupdate", () => {
                if (!this.pitchTriggered && this.videoElement.currentTime >= this.options.pitchTimeSeconds) {
                    this.triggerPitch();
                }
            });
        }
        
        // TODO: Add listeners for custom controls if hideControls is true
        // TODO: Add listeners for tracking metrics (views, completion)
    }

    resumeVideo() {
        if (this.videoElement) {
            this.videoElement.play();
        }
    }

    restartVideo() {
        if (this.videoElement) {
            this.videoElement.currentTime = 0;
            this.videoElement.play();
        }
    }

    triggerPitch() {
        this.pitchTriggered = true;
        const pitchElement = document.querySelector(this.options.pitchTargetSelector);
        if (pitchElement) {
            console.log(`VtriusPlayer: Triggering pitch for ${this.options.pitchTargetSelector} at ${this.options.pitchTimeSeconds}s`);
            // Option 1: Remove a hiding class
            pitchElement.classList.remove("vtrius-pitch-hidden");
            // Option 2: Add a visibility class
            pitchElement.classList.add("vtrius-pitch-visible"); 
            // Ensure your CSS handles one of these classes to show the element
        } else {
            console.warn(`VtriusPlayer Warning: Pitch target element '${this.options.pitchTargetSelector}' not found.`);
        }

        // Option 3: Display configured button
        if (this.options.pitchButtonText && this.options.pitchButtonUrl) {
            this.showPitchButton();
        }
    }

    showPitchButton() {
        // Create button overlay if it doesn't exist
        if (!this.pitchButtonOverlay) {
            this.pitchButtonOverlay = document.createElement("div");
            this.pitchButtonOverlay.style.position = "absolute";
            this.pitchButtonOverlay.style.bottom = "20px"; // Position at the bottom
            this.pitchButtonOverlay.style.left = "50%";
            this.pitchButtonOverlay.style.transform = "translateX(-50%)";
            this.pitchButtonOverlay.style.zIndex = "30"; // Above other elements
            this.pitchButtonOverlay.style.display = "none"; // Hidden initially

            const button = document.createElement("a");
            button.href = this.options.pitchButtonUrl;
            button.textContent = this.options.pitchButtonText;
            button.target = "_blank"; // Open in new tab
            // Basic styling, can be customized via CSS classes
            button.style.padding = "12px 25px";
            button.style.backgroundColor = "#2F4F2F"; // Dark Moss Green
            button.style.color = "white";
            button.style.textDecoration = "none";
            button.style.borderRadius = "5px";
            button.style.fontSize = "1.1em";
            button.style.boxShadow = "0 2px 5px rgba(0,0,0,0.3)";

            this.pitchButtonOverlay.appendChild(button);
            this.container.appendChild(this.pitchButtonOverlay);
        }
        // Show the button
        this.pitchButtonOverlay.style.display = "block";
        console.log(`VtriusPlayer: Displaying pitch button: '${this.options.pitchButtonText}' -> ${this.options.pitchButtonUrl}`);
    }

    // --- Public API Methods (Optional) ---
    play() {
        this.videoElement?.play();
    }

    pause() {
        this.videoElement?.pause();
    }

    seek(timeSeconds) {
        if (this.videoElement) {
            this.videoElement.currentTime = timeSeconds;
        }
    }
}

// Example initialization (This part would likely be generated by the embed script)
/*
document.addEventListener("DOMContentLoaded", () => {
    // Fetch options dynamically based on video ID or embed script parameters
    const playerOptions = {
        videoId: 123, 
        videoUrl: "/path/to/video.mp4", // Replace with actual URL
        hideControls: false,
        autoplay: false,
        overlayImageUrl: null, // "/path/to/overlay.png",
        overlayOpacity: 0.8,
        pauseCoverImageUrl: null, // "/path/to/pause.png",
        pauseCoverText: "Continuar assistindo?",
        pitchTimeSeconds: 10, // Show pitch after 10 seconds
        pitchTargetSelector: "#my-elementor-pitch-container" // CSS selector for the container
    };
    new VtriusPlayer("vtrius-player-container-123", playerOptions);
});
*/

// Function to initialize players based on data attributes in placeholder divs
function initVtriusPlayers() {
    const playerPlaceholders = document.querySelectorAll("div[data-vtrius-id]");
    playerPlaceholders.forEach(placeholder => {
        const videoId = placeholder.dataset.vtriusId;
        const containerId = `vtrius-player-container-${videoId}`;
        placeholder.id = containerId; // Assign unique ID to the placeholder

        // TODO: Fetch video options (URL, settings) via AJAX using videoId
        // For now, using placeholder data
        console.log(`Initializing player for video ID: ${videoId}`);
        const playerOptions = {
            videoId: videoId,
            videoUrl: `/placeholder/video_${videoId}.mp4`, // *** Replace with actual dynamic URL fetching ***
            // Fetch other settings (hideControls, overlay, etc.) via AJAX
            hideControls: placeholder.dataset.hideControls === "true",
            autoplay: placeholder.dataset.autoplay === "true",
            overlayImageUrl: placeholder.dataset.overlayImage || null,
            overlayOpacity: parseFloat(placeholder.dataset.overlayOpacity || 1.0),
            pauseCoverImageUrl: placeholder.dataset.pauseCover || null,
            pauseCoverText: placeholder.dataset.pauseText || "Você já estava assistindo...", // Needs i18n
            pitchTimeSeconds: placeholder.dataset.pitchTime ? parseInt(placeholder.dataset.pitchTime) : null,
            pitchTargetSelector: placeholder.dataset.pitchTarget || null
        };

        new VtriusPlayer(containerId, playerOptions);
    });
}

// Initialize players when the script loads or DOM is ready
document.addEventListener("DOMContentLoaded", initVtriusPlayers);

