-- /home/ubuntu/vtrius_web_system/db/schema.sql
-- Schema for the Vtrius Web System Database (SQLite)

-- Users Table (Adapted from licenses)
CREATE TABLE IF NOT EXISTS users (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    username TEXT UNIQUE NOT NULL,
    email TEXT UNIQUE NOT NULL,
    password_hash TEXT NOT NULL,
    status TEXT NOT NULL DEFAULT 'active', -- 'active', 'inactive', 'expired'
    expiry_date DATETIME, -- NULL for permanent access (e.g., admin)
    is_admin INTEGER DEFAULT 0, -- 1 for admin, 0 for regular user
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
);

-- Trigger to update 'updated_at' timestamp on user update
CREATE TRIGGER IF NOT EXISTS update_user_updated_at
AFTER UPDATE ON users
FOR EACH ROW
BEGIN
    UPDATE users SET updated_at = CURRENT_TIMESTAMP WHERE id = OLD.id;
END;

-- Add indexes for faster lookups
CREATE INDEX IF NOT EXISTS idx_user_username ON users (username);
CREATE INDEX IF NOT EXISTS idx_user_email ON users (email);
CREATE INDEX IF NOT EXISTS idx_user_status ON users (status);

-- Videos Table
CREATE TABLE IF NOT EXISTS videos (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    user_id INTEGER NOT NULL,
    title TEXT NOT NULL,
    original_filename TEXT,
    storage_path TEXT UNIQUE NOT NULL, -- Relative path within uploads/videos/USER_ID/
    mime_type TEXT,
    filesize INTEGER,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

CREATE INDEX IF NOT EXISTS idx_video_user_id ON videos (user_id);

-- Video Settings Table
CREATE TABLE IF NOT EXISTS video_settings (
    video_id INTEGER PRIMARY KEY,
    hide_controls INTEGER DEFAULT 0, -- 1 to hide
    overlay_image_path TEXT, -- Relative path within uploads/overlays/USER_ID/
    overlay_opacity REAL DEFAULT 1.0, -- 0.0 to 1.0
    pause_cover_image_path TEXT, -- Relative path within uploads/pause_covers/USER_ID/
    pause_cover_text TEXT,
    pitch_time_seconds INTEGER, -- Time in seconds for pitch content to appear
    pitch_target_selector TEXT, -- CSS selector for the pitch container (e.g., '#my-pitch-div')
    pitch_button_text TEXT, -- Text for the button shown at pitch time
    pitch_button_url TEXT, -- URL for the button shown at pitch time
    autoplay INTEGER DEFAULT 0, -- 1 for autoplay (added)
    FOREIGN KEY (video_id) REFERENCES videos(id) ON DELETE CASCADE
);

-- A/B Tests Table
CREATE TABLE IF NOT EXISTS ab_tests (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    user_id INTEGER NOT NULL,
    name TEXT NOT NULL,
    main_video_id INTEGER NOT NULL, -- Control video
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (main_video_id) REFERENCES videos(id) ON DELETE CASCADE
);

CREATE INDEX IF NOT EXISTS idx_abtest_user_id ON ab_tests (user_id);

-- A/B Test Variations Table
CREATE TABLE IF NOT EXISTS ab_variations (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    ab_test_id INTEGER NOT NULL,
    video_id INTEGER NOT NULL,
    is_control INTEGER DEFAULT 0, -- 1 if this is the main_video_id (control)
    FOREIGN KEY (ab_test_id) REFERENCES ab_tests(id) ON DELETE CASCADE,
    FOREIGN KEY (video_id) REFERENCES videos(id) ON DELETE CASCADE
);

CREATE INDEX IF NOT EXISTS idx_abvariation_test_id ON ab_variations (ab_test_id);

-- A/B Test Statistics Table (Example - might need refinement)
CREATE TABLE IF NOT EXISTS ab_stats (
    variation_id INTEGER NOT NULL,
    date DATE NOT NULL,
    views INTEGER DEFAULT 0,
    completions INTEGER DEFAULT 0, -- e.g., watched >= 90%
    PRIMARY KEY (variation_id, date),
    FOREIGN KEY (variation_id) REFERENCES ab_variations(id) ON DELETE CASCADE
);


